<?php

namespace Raptor\Headless;

use Raptor\ACF\Utils;
use Raptor\ACF\Field_Types;

class Redirects {
    function __construct() {
        add_action( 'acf/init', [ $this, 'acf' ], 15 );
        add_filter( 'acf/validate_value', [ $this, 'validate_redirects' ], 10, 3 );
        add_action( 'graphql_register_types', [ $this, 'graphql_register_types' ] );
        add_filter( 'raptor/acf/settings/redirects', [ $this, 'load_original_redirects_field' ] );
    }

    /**
     * Register ACF fields.
     */
    function acf() {
        Utils\add_settings_tab([
            'group' => 'redirects',
            'label' => 'Redirects',
            'fields' => [
                Field_Types\repeater_field(
                    'Redirects',
                    [
                        'sub_fields' => [
                            Field_Types\text_field(
                                'Source'
                            ),
                            Field_Types\text_field(
                                'Destination'
                            )
                        ],
                        'instructions' => 'Only relative URLs are supported, eg. <code>/about/</code>',
                        'button_label' => 'Add redirect'
                    ]
                )
            ]
        ]);

        Utils\register_settings_group([
            'name' => 'redirects',
            'title' => 'Redirects',
            'menu_title' => 'Redirects',
            'redirect' => false,
            'show_in_graphql' => false,
            'parent_slug' => 'raptor-settings'
        ]);
    }


    /**
     * Validate redirect fields to ensure noly relative URLs are passed.
     * 
     * @param mixed $valid 
     * @param mixed $value
     * @param array $field
     */
    function validate_redirects( $valid, $value, $field ) {
        if ( $field['parent'] === 'repeater_redirects' ) {
            if ( stripos( $value, 'http' ) === 0 ) {
                return 'Only relative URLs are supported.';
            }

            if ( stripos( $value, '/' ) !== 0 ) {
                return 'Relative URLs must start with a forward slash "/"';
            }
        }

        return $valid;
    }
    

    function graphql_register_types() {
        register_graphql_object_type( 'Redirect', [
            'description' => 'A redirect',
            'fields' => [
                'source' => ['type' => 'String'],
                'destination' => ['type' => 'String'],
            ],
        ]);
        
    
        register_graphql_field( 'RootQuery', 'redirectsJson', [
            'type' => [ 'list_of' => 'Redirect' ],
            'resolve' => function() {
                $redirects = [];
    
                $redirects_csv_file = get_field( 'csv-file', 'option' );
    
                if ( $redirects_csv_file ) {
                    $file_path = get_attached_file( $redirects_csv_file['ID'] );
    
                    if ( ( $handle = fopen( $file_path, "r" ) ) !== FALSE ) {
                        while ( ( $data = fgetcsv( $handle, 1000, "," ) ) !== FALSE ) {
        
                            $redirects[] = [
                                'source' => $data[0],
                                'destination' => $data[1],
                            ];
                        }
                        fclose( $handle );
                    }
                } else {
                    $redirects = [];
                    $redirects_group = get_field( 'redirects', 'option' );
                    $redirects_new = $redirects_group ? $redirects_group['redirects'] : false;
                    $redirects_org = get_field( 'redirects-v2', 'option' );

                    if ( $redirects_new && count( $redirects_new ) > 0 ) {
                        $redirects = $redirects_new;
                    }

                    if ( $redirects_org && count( $redirects_org ) > 0 ) {
                        if ( !empty( $redirects ) ) {
                            $redirects = array_merge( $redirects, $redirects_org );
                        } else {
                            $redirects = $redirects_org;
                        }
                    }
                }
    
                return !empty( $redirects ) ? $redirects : null;
            }
        ]);
    }

    function load_original_redirects_field( $fields ) {
        $fields[] = Field_Types\repeater_field(
            'Redirects v2',
            [
                'sub_fields' => [
                    Field_Types\text_field(
                        'Source'
                    ),
                    Field_Types\text_field(
                        'Destination'
                    )
                ],
                'instructions' => 'Deprecated. Please use the fields above.',
                'button_label' => 'Add redirect'
            ]
        );
    
        return $fields;
    }
}

new Redirects;
