<?php
namespace Raptor\Forms;

use Raptor\Forms\Form;

require_once( RAPTOR_INC . '/forms/handler.php' );
require_once( RAPTOR_INC . '/forms/form.php' );
require_once( RAPTOR_INC . '/forms/field-type.php' );
require_once( RAPTOR_INC . '/forms/graphql.php' );
require_once( RAPTOR_INC . '/forms/export-submissions.php' );

// Field types
require_once( RAPTOR_INC . '/forms/field-types/text.php' );
require_once( RAPTOR_INC . '/forms/field-types/email.php' );
require_once( RAPTOR_INC . '/forms/field-types/tel.php' );
require_once( RAPTOR_INC . '/forms/field-types/textarea.php' );
require_once( RAPTOR_INC . '/forms/field-types/select.php' );
require_once( RAPTOR_INC . '/forms/field-types/checkbox.php' );
require_once( RAPTOR_INC . '/forms/field-types/radio.php' );
require_once( RAPTOR_INC . '/forms/field-types/date.php' );
require_once( RAPTOR_INC . '/forms/field-types/time.php' );
require_once( RAPTOR_INC . '/forms/field-types/number.php' );
require_once( RAPTOR_INC . '/forms/field-types/url.php' );
require_once( RAPTOR_INC . '/forms/field-types/file.php' );
require_once( RAPTOR_INC . '/forms/field-types/submit.php' );


class Forms {
    /** @var Handler $handler */
    var $handler;

    /** @var Form[] $forms */
    var $forms = [];


    /**
     * Class hooks and assign handler.
     */
    function __construct() {
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_scripts' ], 10 );
        add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_frontend_scripts' ] );
        add_action( 'raptor_forms_render_field_before', [ $this, 'recaptcha_notice' ], 10, 2 );

        add_action( 'wp_ajax_raptor_form_save', [ $this, 'save_handler' ] );
        add_action( 'wp_ajax_raptor_form_retrieve', [ $this, 'retrieve_handler' ] );
        add_filter( 'raptor_js_object', [ $this, 'raptor_js_object' ] );

        add_action( 'init', function() {
            $handler_class = apply_filters( 'raptor/forms/handler/class', 'Raptor\Forms\Handler' );
            $this->handler = new $handler_class;
        });
    }


    /**
     * Enqueue the form builder JavaScript file
     */
    function enqueue_scripts() {
        wp_register_script( 'raptor-react', raptor_get_asset_url( 'assets/js/form-builder.js' ), [ 'raptor-admin-script' ], raptor()->version, [ 'in_footer' => true ] );

        $screen = get_current_screen();

        if ( $screen->base == 'post' && $screen->post_type == 'raptor_form' ) {
            wp_enqueue_script( 'raptor-react' );
        }
    }


    function enqueue_frontend_scripts() {
        /**
         * Raptor Toolkit does not enqueue this script, that is handled by JavaScript in Raptor Frontend
         */
        if ( raptor_recaptcha_enabled() ) {
            $recaptcha_settings = raptor_get_settings( 'forms', 'recaptcha' );
            $recaptcha_site_key = ( isset( $recaptcha_settings['site_key'] ) && !empty( $recaptcha_settings['site_key'] ) ) ? $recaptcha_settings['site_key'] : false;

            if ( $recaptcha_site_key ) {
                wp_register_script( 'raptor-recaptcha', "https://www.google.com/recaptcha/api.js?render=$recaptcha_site_key", [ 'raptor-main' ], raptor()->version, [ 'in_footer' => true ] );
            }
        }
    }


    /**
     * Save the form data via AJAX request
     */
    function save_handler() {
        $post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;

        $fields_data = isset( $_POST['raptor'] ) ? $_POST['raptor'] : [];
        $fields_data = json_decode( str_replace( "\\", '', $fields_data ), true );

        $groups_data = isset( $_POST['groups'] ) ? $_POST['groups'] : [];
        $groups_data = json_decode( str_replace( "\\", '', $groups_data ), true );

        $settings_data = isset( $_POST['settings'] ) ? $_POST['settings'] : [];
        $settings_data = json_decode( str_replace( "\\", '', $settings_data ), true );

        foreach ( $settings_data as $setting_key => $value ) {
            update_post_meta( $post_id, $setting_key, $value );
        }

        update_post_meta( $post_id, 'fields', $fields_data );
        update_post_meta( $post_id, 'groups', $groups_data );

        wp_send_json([
            'success' => true
        ]);
    }


    /**
     * Get the form data via AJAX request
     */
    function retrieve_handler() {
        $post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;

        $settings = [
            'recepients' => get_post_meta( $post_id, 'recepients', true ),
            'subject' => get_post_meta( $post_id, 'subject', true ),
            'reply_to' => get_post_meta( $post_id, 'reply_to', true ),
            'redirect' => get_post_meta( $post_id, 'redirect', true ),
            'recaptcha' => get_post_meta( $post_id, 'recaptcha', true ),
            'mailchimp_send' => get_post_meta( $post_id, 'mailchimp_send', true ),
            'mailchimp_tag' => get_post_meta( $post_id, 'mailchimp_tag', true ),
            'data_capture' => (bool) get_post_meta( $post_id, 'data_capture', true ),
            'mail_send' => (bool) get_post_meta( $post_id, 'mail_send', true )
        ];

        wp_send_json([
            'settings' => $settings,
            'fields' => get_post_meta( $post_id, 'fields', true ),
            'groups' => get_post_meta( $post_id, 'groups', true )
        ]);
    }


    /**
     * Display the reCAPTCHA text before the submit button on each form.
     * 
     * @see https://developers.google.com/recaptcha/docs/faq#id-like-to-hide-the-recaptcha-badge.-what-is-allowed
     * 
     * @param array $field
     * @param Form $form
     */
    function recaptcha_notice( array $field, Form $form ) {
        $text = 'This site is protected by reCAPTCHA and the Google <a href="https://policies.google.com/privacy">Privacy Policy</a> and <a href="https://policies.google.com/terms">Terms of Service</a> apply.';

        if ( $form->recaptcha_enabled() && $field['type'] == 'submit' ) {
            printf( '<p class="recaptcha-notice">%s</p>', $text );
        }
    }


    function raptor_js_object( array $object ) {
        $forms = [];
        $query = new \WP_Query([
            'post_type' => 'raptor_form',
            'posts_per_page' => -1,
            'fields' => 'ids'
        ]);

        if ( $query->found_posts ) {
            foreach ( $query->posts as $post_id ) {
                $form = new Form( $post_id );
                $forms[ $form->name ] = [
                    'fields' => format_conditional_logic( $form->fields ),
                    'groups' => format_conditional_logic( $form->groups )
                ];
            }
        }

        $object['forms'] = $forms;

        return $object;
    }
}

raptor()->forms = new Forms;


global $raptor_forms_rendered;

$raptor_forms_rendered = [];


global $raptor_form_core;

$raptor_form_core = [];


// Register the Raptor forms post type
$raptor_form_core['form'] = new \Raptor_Custom_Post_Type(
    [
        'label'         => 'Form',
        'name'          => 'raptor_form',
        'menu_icon'     => 'dashicons-email-alt'
    ],
    [
        'show_ui'       => true,
        'has_archive'   => false,
        'map_meta_cap'  => true,
        'supports'      => [ 'title' ]
    ]
);


// Register the Raptor forms post type
$raptor_form_core['data'] = new \Raptor_Custom_Post_Type(
    [
        'label'         => 'Form Submissions',
        'name'          => 'raptor_form_data',
        'plural'        => false
    ],
    [
        'show_ui'       => true,
        'show_in_menu'  => 'edit.php?post_type=raptor_form',
        'has_archive'   => false,
        'capabilities'  => [
            'create_posts'       => 'do_not_allow',
            'edit_post'          => 'update_core',
            'read_post'          => 'update_core',
            'delete_post'        => 'update_core',
            'edit_posts'         => 'update_core',
            'edit_others_posts'  => 'update_core',
            'delete_posts'       => 'update_core',
            'publish_posts'      => 'update_core',
            'read_private_posts' => 'update_core'
        ],
        'map_meta_cap'  => true,
        'supports'      => [ 'title', 'custom-fields' ]
    ]
);


add_filter( 'raptor_custom_post_type_register_args', function( $args, $post_type ) {
    if ( $post_type == 'raptor_form_data' ) {
        $args['labels']['all_items'] = 'Submissions';
        $args['labels']['edit_item'] = 'View Submission';
    }

    return $args;
}, 10, 2 );


add_filter( 'manage_raptor_form_data_posts_columns', function( $columns ) {
    unset( $columns['title'] );
    unset( $columns['date'] );

    $columns['data_id']         = 'Submission ID';
    $columns['data_date']       = 'Date';
    $columns['data_from_form']  = 'Form';

    return $columns;
});


add_filter( 'manage_edit-raptor_form_data_sortable_columns', function( $columns ) {
    $columns['data_id']     = 'ID';
    $columns['data_date']   = 'date';

    return $columns;
});


add_action( 'manage_raptor_form_data_posts_custom_column', function( $column, $post_id ) {
    if ( $column == 'data_id' ) {
        $title = strval( $post_id );

        echo '<a href="' . esc_url( admin_url( 'post.php?post=' . $post_id . '&action=edit' ) ) . '"><strong>#' . esc_html( $title ) . '</strong></a>';
    }

    if ( $column == 'data_date' ) {

        $date = get_the_date( 'jS M Y \@ H:i' );

        echo $date;
    }

    if ( $column == 'data_from_form' ) {
        $form_id = get_post_meta( $post_id, 'form_id', true );
        $form = raptor_get_form_instance( $form_id );

        if ( $form_id > 0 ) {
            echo $form->title;

        } else {
            echo 'Unknown';
        }
    }

}, 10, 2 );


add_action( 'pre_get_posts', function( $query ) {
    if ( !is_admin() ) return;

    $orderby = $query->get( 'orderby' );

    if ( $orderby == 'data_id' ) {
        $query->set( 'meta_key', 'post_id' );
        $query->set( 'orderby', 'meta_value_num' );
    }
});


add_filter( 'post_row_actions', function( $actions, $post ) {
    if ( $post->post_type == 'raptor_form' ) {
        unset( $actions['trash'] );
    }

    if ( $post->post_type == 'raptor_form_data' ) {
        unset( $actions['inline hide-if-no-js'] );

        $actions['edit'] = '<a href="' . esc_url( admin_url( 'post.php?post=' . $post->ID . '&action=edit' ) ) . '">Edit</a>';
    }

    return $actions;
}, 100, 2 );


add_filter( 'manage_raptor_form_posts_columns', function( $columns ) {
    return array_merge(
        array_slice( $columns, 0, 2 ),
        [ 'form_html_id' => 'HTML ID' ],
        array_slice( $columns, 2 )
    );
});


add_action( 'manage_raptor_form_posts_custom_column', function( $column, $post_id ) {
    if ( $column == 'form_html_id' ) {
        printf( '<span class="raptor-copy-form-html-id"><pre><code>raptor-form-%d</code></pre><svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 -960 960 960"><path d="M364.615-280Q337-280 318.5-298.5 300-317 300-344.615v-430.77Q300-803 318.5-821.5 337-840 364.615-840h310.77Q703-840 721.5-821.5 740-803 740-775.385v430.77Q740-317 721.5-298.5 703-280 675.385-280h-310.77Zm0-40h310.77q9.231 0 16.923-7.692Q700-335.385 700-344.615v-430.77q0-9.23-7.692-16.923Q684.616-800 675.385-800h-310.77q-9.23 0-16.923 7.692Q340-784.615 340-775.385v430.77q0 9.23 7.692 16.923Q355.385-320 364.615-320Zm-120 160Q217-160 198.5-178.5 180-197 180-224.615v-470.77h40v470.77q0 9.231 7.692 16.923Q235.384-200 244.615-200h350.77v40h-350.77ZM340-320v-480 480Z"/></svg></span>', $post_id );
    }
}, 10, 2 );


add_filter( 'bulk_actions-edit-raptor_form', function( $actions ) {
    unset( $actions['trash'] );

    return $actions;
}, 100, 1 );


// Admin Stuff

function raptor_form_data_meta_boxes() {
    global $post;

    remove_meta_box( 'slugdiv', 'raptor_form_data', 'normal' );
    remove_meta_box( 'submitdiv', 'raptor_form_data', 'side' );

    add_meta_box(
        'raptor-form-builder',
        'Form Builder',
        'Raptor\Forms\raptor_form_builder',
        'raptor_form',
        'normal'
    );

    add_meta_box(
        'raptor-form-data-content',
        'Data',
        'Raptor\Forms\raptor_form_data_get_content',
        'raptor_form_data',
        'normal'
    );

    if ( !get_post_meta( $post->ID, 'raptor_form_mail_error', true ) ) return;

    add_meta_box(
        'raptor-form-data-error',
        'Mail Error',
        'Raptor\Forms\raptor_form_get_error',
        'raptor_form_data',
        'normal'
    );
}
add_action( 'add_meta_boxes', 'Raptor\Forms\raptor_form_data_meta_boxes' );


add_action( 'edit_form_top', function( $post ) {
    if ( $post->post_type == 'raptor_form_data' ) {
        ?>
        <style type="text/css"> #post-body-content { display: none; } </style>
        <?php
    }
} );


function raptor_form_data_get_content() {
    global $post, $raptor_forms;

    $values = raptor_get_form_data( $post->ID );
    $form_id = get_post_meta( $post->ID, 'form_id', true );
    $form = raptor_get_form_instance( $form_id );


    list( $object_type, $object_id ) = explode( '-', get_post_meta( $post->ID, 'source', true ) );

    if ( $object_type == 'post' ) {
        $source_object = get_post( $object_id );
        $source_title = $source_object->post_title;
        $source_link = get_permalink( $object_id );

    } else if ( $object_type == 'term' ) {
        $source_object = get_term( $object_id );
        $source_title = $source_object->name;
        $source_link = get_permalink( $object_id );
    }

    $summary = sprintf( 'Submitted using the <strong>%s</strong> form via <a href="%s">%s</a> on %s', $form->title, $source_link, $source_title, get_the_date( 'l jS M Y \@ H:i' ) );
    ?>

    <div>
        <h2 class="raptor-meta-box--heading">#<?php echo esc_html( $post->ID ); ?></h2>
        <span><?php echo $summary; ?></span>
        <button type="button" class="button" name="raptor_form_submission_resend">Re-send email</button>
    </div>
    
    <table class="raptor-form-data-content-table" border="0" cellspacing="0" cellpadding="0">
        <tbody>
            <?php

            $key = 0;

            foreach ( $values as $field_name => $value ) :
                $field = $form->fields[ $key ];

                if ( $field['type'] === 'submit' ) {
                    continue;
                }

                $value = wp_unslash( $value );

                if ( $field['type'] === 'checkbox' && in_array( $value, [ 'on', '' ] ) ) {
                    $value = $value === 'on' ? 'Yes' : 'No';
                }
                ?>
            <tr>
                <td><?php echo esc_html( $field['label'] ); ?></td>
                <td><?php echo esc_html( $value ); ?></td>
            </tr>
            <?php $key++; endforeach; ?>
        </tbody>
    </table>

    <?php
}


function raptor_form_get_error() {
    global $post;

    $error = get_post_meta( $post->ID, 'raptor_form_mail_error', true );

    echo '<pre><code>';

    print_r( esc_html( maybe_unserialize( base64_decode( $error ) ) ) );

    echo '</code></pre>';
}


function raptor_form_builder() {
    echo '<div id="__form_builder_root"></div>';
}

/**
 * Get the accepted file types for a file input
 * 
 * @since 1.4
 * @param array $field The field 
 * @param string $format The format to return, regex or attribute
 * @return string The list of file types in the requested format
 */
function raptor_form_get_file_types( $field, $format = 'regex' ) {
    $file_types = [];
    $formatted_file_types = [];

    $default_file_types = [
        'png',
        'jpg',
        'jpeg',
        'gif',
        'pdf',
        'doc',
        'docx',
        'ppt',
        'pptx',
        'pages',
        'keynote'
    ];

    if ( empty( $field['accept'] ) ) {
        $file_types = $default_file_types;

    } else {
        $file_types = explode( ',', raptor_form_get_file_accept( $field['accept'] ) );
    }

    if ( is_array( $file_types ) ) {

        foreach ( $file_types as $type ) {

            if ( is_string( $type ) ) {
                $type = preg_split( '/[\s,]+/', $type );
            }

            $formatted_file_types = array_merge( $formatted_file_types, (array) $type );
        }

        $formatted_file_types = array_unique( array_filter( $formatted_file_types ) );
    }

    $output = '';

    foreach ( $formatted_file_types as $type ) {
        $type = trim( $type, ' ,|' );

        if ( $format === 'attr' ) {
            if ( preg_match( '/^application/', $type ) ) {
                $output .= sprintf( '%s,', $type );
            } else {
                $output .= sprintf( '.%s,', $type );
            }
        } else {
            $output .= preg_replace(
                [
                    '/\//',
                    '/\./'
                ],
                [
                    '\/',
                    '\.'
                ],
                $type
            );
            
            $output .= '|';
        }
    }

    return trim( $output, ' ,|' );
}


/**
 * Get the unique file type specifier.
 * 
 * @param string  $accept_id The identifier for a collection of file types.
 * @return string The value for the accept attribute
 */
function raptor_form_get_file_accept( string $accept_id, $field = null ) {
    $value = '';

    switch ( $accept_id ) {
        case 'document':
            $value = '.pdf,.doc,.docx,application/msword,application/vnd.openxmlformats-officedocument.wordprocessingml.document';
            break;

        case 'spreadsheet':
            $value = '.xlsx,.xls,.csv,application/vnd.openxmlformats-officedocument.spreadsheetml.sheet';
            break;

        case 'image':
            $value = 'image/*';
            break;

        case 'video':
            $value = 'video/*';
            break;

        case 'audio':
            $value = 'audio/*';
            break;
    }

    return apply_filters( 'raptor_form_get_file_accept', $value, $field );
}

/**
 * Get the fields of a form.
 * 
 * @param int $form_id The post ID of the form.
 * @return array|false
 */
function raptor_get_form_fields( int $form_id ) {
    return get_post_meta( $form_id, 'fields', true );
}


/**
 * Get the data for a form submission.
 * 
 * @param int $data_id The post ID of the form submission
 * @return array 
 */
function raptor_get_form_data( int $data_id ) {
    /**
     * Security measure, form submission data should never be fetchable from the frontend
     */
    if ( apply_filters( 'raptor_get_form_data_abort', ( !is_admin() && !class_exists( 'WP_CLI' ) || !is_admin() && !wp_doing_cron() ) ) ) {
        trigger_error( 'Attempted to get form submission data during a potentially unsafe request. Check that how you are calling this function complies with security best practices, as you may be exposing data to the public.' );

        return [];
    }

    $form_id = get_post_meta( $data_id, 'form_id', true );
    $form = raptor_get_form_instance( $form_id );

    $data = [];

    foreach ( $form->fields as $field ) {
        if ( $field['type'] == 'submit' || $field['type'] == 'file' ) {
            continue;
        }

        $data[ $field['name'] ] = get_post_meta( $data_id, $field['name'], true );
    }

    // Remove empty values.
    return $data;
}


/**
 * Allow submissions to be filtered by form
 * 
 * @param string $post_type
 */
function raptor_forms_form_filter_dropdown( $post_type ) {
    if ( $post_type == 'raptor_form_data' ) {
        $options = array_map( function( $form_post ) {
            return [
                'id' => $form_post->ID,
                'title' => $form_post->post_title
            ];
        }, get_posts( [ 'post_type' => 'raptor_form' ] ) );

        echo '<label class="screen-reader-text" for="form-filter">Filter by form</label>';
        echo '<select name="filter_raptor_form" id="form-filter">';
        echo '<option value="0">All forms</option>';

        foreach ( $options as $option ) {
            printf( '<option value="%s" %s>%s</option>', $option['id'], selected( $option['id'], get_query_var( 'filter_raptor_form' ), false ), $option['title'] );
        }

        echo '</select>';
    }
}
add_action( 'restrict_manage_posts', 'Raptor\Forms\raptor_forms_form_filter_dropdown' );


/**
 * Set the _form_name meta key to be queryable in admin.
 * 
 * @param WP_Query $query
 */
function raptor_forms_set_query( $query ) {
    if ( is_admin() && $query->is_main_query() && get_query_var( 'filter_raptor_form' ) ) {
        $query->set( 'meta_key', 'form_id' );
        $query->set( 'meta_value', get_query_var( 'filter_raptor_form' ) );
    }
}
add_action( 'pre_get_posts', 'Raptor\Forms\raptor_forms_set_query' );


/**
 * Add the form query var.
 * 
 * @param array $vars
 */
function raptor_forms_add_form_var( $vars ) {
    $vars[] = 'filter_raptor_form';

    return $vars;
}
add_filter( 'query_vars', 'Raptor\Forms\raptor_forms_add_form_var' );


/**
 * @param array|string $things Fields or groups
 * @return array|string
 */
function format_conditional_logic( $things ) {
    if ( is_array( $things ) ) {
        foreach ( $things as &$thing ) {
            // phpcs:ignore Generic.Formatting.MultipleStatementAlignment.NotSame
            if ( isset( $thing['conditional_logic'] ) &&
                is_array( $thing['conditional_logic'] )
            ) {
                $thing['conditional_logic'] = array_map(
                    function( $and ) {
                        return array_values( $and );
                    },
                    array_values(
                        $thing['conditional_logic']
                    )
                );
            }
        }
    }

    return $things;
}
