<?php
/**  
 *  Output the font associated link tags
 *
 *  @since 1.0
 */
function raptor_fonts() {
    if ( !empty( raptor()->fonts ) ) {
        $config = raptor()->fonts;
        $preconnect = !empty( $config['preconnect'] ) ? $config['preconnect'] : [];
        $web = !empty( $config['web'] ) ? $config['web'] : [];
        $custom = !empty( $config['custom'] ) ? $config['custom'] : [];
        $all_fonts = array_merge( $web, $custom );

        $preload = array_map( function( $font ) {
            return $font['file'];
        }, $all_fonts );

        raptor_preload_config( $preconnect, $preload );
    }
}
add_action( 'wp_head', 'raptor_fonts', 11 );


/**
 * Output the relevant HTML tags for preloading fonts
 * 
 * @param array $preconnect
 * @param array $preload
 */
function raptor_preload_config( array $preconnect = [], array $preload = [] ) {
    if ( is_array( $preconnect ) && $preconnect ) {
        foreach ( $preconnect as $href ) {
            $preconnect_attr = [
                'rel'           => 'preconnect',
                'href'          => $href,
                'crossorigin'   => ''
            ];
            echo '<link ' . raptor_output_attr( $preconnect_attr, true ) . '>';
        }
    }

    if ( is_array( $preload ) && $preload ) {
        foreach ( $preload as $href ) {
            $preload_attr = [
                'rel'       => 'preload',
                'href'      => $href,
                'as'        => 'style'
            ];
            $async_attr = [
                'rel'       => 'stylesheet',
                'href'      => $href,
                'media'     => 'print',
                'onLoad'    => "this.media='all'"
            ];
            echo '<link ' . raptor_output_attr( $preload_attr, true ) . '>';
            echo '<link ' . raptor_output_attr( $async_attr, true ) . '>';
        }
        echo '<noscript>';
        foreach ( $preload as $href ) {
            $noscript_attr = [
                'rel'       => 'stylesheet',
                'href'      => $href
            ];
            echo '<link ' . raptor_output_attr( $noscript_attr, true ) . '>';
        }
        echo '</noscript>';
    }
}


function raptor_load_fonts() {
    if ( !empty( raptor()->fonts ) ) {
        $config = raptor()->fonts;
        $web = !empty( $config['web'] ) ? $config['web'] : [];
        $custom = !empty( $config['custom'] ) ? $config['custom'] : [];
        $all_fonts = array_merge( $web, $custom );
        $font_names = [];

        foreach ( $all_fonts as $font ) {
            is_array( $font['name'] ) ? $font_names = array_merge( $font_names, $font['name'] ) : array_push( $font_names, $font['name'] );
        }
        ?>
<script id="raptor-load-fonts">
    ( () => {
        const fontNames = <?php echo json_encode( $font_names ); ?>;
        const hasFonts = fontNames && Boolean(fontNames.length);
        const apiAvailable = "fonts" in document;

        function handleLoadComplete() {
            document.documentElement.classList.add( 'fonts-loaded' );
        }

        function handleFontLoad(fontFaces) {
            fontFaces.forEach((fontFace) => {
                addClassName(fontFace.family);
            })
        }

        function fontMapper(fontName) {
            return document.fonts
                .load(`1rem "${fontName}"`)
                .then(handleFontLoad)
                .catch(errorFallback);
        }

        function loadFonts() {
            const fonts = fontNames.map(fontMapper);
            Promise.all(fonts).then(handleLoadComplete).catch(errorFallback);
        }

        function errorFallback() {
            fontNames.forEach(addClassName);
        }

        function kebabCase( str ) {
            return str
                .match(/[A-Z]{2,}(?=[A-Z][a-z0-9]*|\b)|[A-Z]?[a-z0-9]*|[A-Z]|[0-9]+/g)
                .filter(Boolean)
                .map(x => x.toLowerCase())
                .join("-");
        }

        function addClassName( fontName ) {
            document.documentElement.classList.add(`wf-${kebabCase(fontName)}`);
        }

        if (!apiAvailable) {
            console.info(`document.fonts API error: Font loading API not available`);
            console.info(`Replacing fonts instantly. FOUT handling failed.`);
            errorFallback();
            return;
        }

        if (hasFonts && apiAvailable) {
            loadFonts();
        }
    })();
</script>
        <?php
    }
}
add_action( 'wp_footer', 'raptor_load_fonts', 5 );
