<?php
namespace Raptor;

use Raptor\API;
/**
 * Update mechanism.
 */
class Updater {
    function __construct() {
        add_filter( 'pre_set_site_transient_update_plugins', [ $this, 'modify_update_plugins' ] );
        /**
         * Local Raptor API overrides.
         */
        if ( defined( 'RAPTOR_API_DEV' ) ) {
            add_filter( 'http_request_host_is_external', [ $this, 'allow_localhost' ], 10, 3 );
            add_filter( 'http_allowed_safe_ports', [ $this, 'allow_localhost_port' ], 10, 3 );
        }
    }


    function get_license_key() {
        return get_option( 'raptor_toolkit_license', true );
    }


    /**
     * Activate the Raptor Toolkit license.
     * 
     * @param string $key
     * @return array|bool The response or false
     */
    function activate_license( string $key = '' ) {
        if ( empty( $key ) && defined( 'RAPTOR_TOOLKIT_LICENSE' ) ) {
            $key = \RAPTOR_TOOLKIT_LICENSE;
        }

        $response = API::post(
            'v1/toolkit/activate',
            [
                'key' => $key,
                'toolkit_version' => raptor()->version,
                'wp_name'     => get_bloginfo( 'name' ),
                'wp_url'      => home_url(),
                'wp_version'  => get_bloginfo( 'version' ),
                'php_version' => PHP_VERSION
            ]
        );

        if ( is_array( $response ) ) {
            if ( $response['success'] ) {
                update_option( 'raptor_toolkit_license', $key );
                delete_site_transient( 'update_plugins' );
            }

            return $response;
        }

        return false;
    }


    function update_check() {
        $query_params = [
            'key' => $this->get_license_key(),
            'version' => raptor()->version,
            'url' => home_url()
        ];

        if ( defined( 'RAPTOR_API_DEV' ) ) {
            $query_params['raptor_api_dev'] = true;
        }

        $response = API::get( 'v2/toolkit/check-update', $query_params );

        return $response;
    }


    function modify_update_plugins( $transient ) {
        $plugin_file = 'raptor-toolkit/raptor-toolkit.php';
        $plugin_slug = 'raptor-toolkit';
        /** @var array|string|\WP_Error */
        $update = $this->update_check();

        if ( is_wp_error( $update ) ) {
            return $transient;
        }

        if ( \gettype( $update ) === 'string' ) {
            return $transient;
        }

        if ( defined( 'RAPTOR_MODE' ) ) {
            $plugin_file = 'toolkit/raptor-toolkit.php';
            $plugin_slug = 'toolkit';
        }

        if ( isset( $update['available'] ) ) {
            unset( $update['available'] );

            $transient->response[ $plugin_file ] = (object) $update;
        }

        return $transient;
    }


    function allow_localhost( bool $external, string $host, string $url ) {
        if ( preg_match( '/v1\/toolkit\/download/', $url ) ) {
            $external = true;
        }

        return $external;
    }


    function allow_localhost_port( array $allowed_ports, string $host, string $url ) {
        $allowed_ports[] = 3010;
        
        return $allowed_ports;
    }
}

new Updater;
