<?php
namespace Raptor;

use WP_CLI;

/**
 * Fleximod.
 */
class Fleximod {
    /** @var string $name The name of the Fleximod */
    var $name = '';

    /** @var array $objects The post types and/or taxonomies to query */
    var $objects = [];

    /** @var callable $callback The function to run the mod on a Flexi block */
    var $callback = null;


    /**
     * Setup the WP CLI command
     * 
     * @param string $name
     * @param array|null $objects Defaults to `page` only.
     * @param callable $callback
     */
    function __construct( string $name, $objects, $callback ) {
        $this->name = $name;
        $this->callback = $callback;

        /**
         * If no objects are set, we default to only `pages`
         */
        if ( !$objects ) {
            $this->objects = [ 'post_types' => [ 'page' ] ];
        } else {
            $this->objects = $objects;
        }

        add_action( 'cli_init', [ $this, 'cli_init' ] );
    }


    /**
     * Setup the WP CLI command
     */
    function cli_init() {
        WP_CLI::add_command( "raptor fleximod {$this->name}", [ $this, 'process' ] ); 
    }


    /**
     * Top level function to call query on posts and/or terms.
     */
    function process() {
        WP_CLI::line( "Starting {$this->name} Fleximod..." );

        if ( isset( $this->objects['post_types'] ) && !empty( $this->objects['post_types'] ) ) {
            $this->process_post_types();
        }

        if ( isset( $this->objects['taxonomies'] ) && !empty( $this->objects['taxonomies'] ) ) {
            $this->process_taxonomies();
        }
    }


    /**
     * Process post types.
     */
    function process_post_types() {
        foreach ( $this->objects['post_types'] as $post_type ) {
            $this->process_post_type( $post_type );
        }
    }


    /**
     * Process an individual post type.
     * 
     * @param string $post_type
     */
    function process_post_type( string $post_type ) {
        $query_args = [
            'post_type' => $post_type,
            'posts_per_page' => -1
        ];

        $query = new \WP_Query( apply_filters( 'raptor/fleximod/post_type_args', $query_args, $post_type, $this->name ) );
        
        if ( $query->have_posts() ) {
            $progress = WP_CLI\Utils\make_progress_bar( "Processing $post_type", $query->found_posts );

            while ( $query->have_posts() ) {
                $query->the_post();
                $progress->tick();

                $flexi = get_field( raptor_get_flexi_field_key(), get_the_ID(), false );

                if ( !$flexi ) {
                    continue;
                }

                foreach ( $flexi as &$flexi_block ) {
                    $flexi_block = call_user_func_array( $this->callback, [ $flexi_block, get_post() ] );
                }

                update_field( raptor_get_flexi_field_key(), array_filter( $flexi ), get_the_ID() );
            }

            $progress->finish();
        } else {
            WP_CLI::warning( "Nothing found in the `$post_type` post type." );
        }
    }


    /**
     * Process taxonomies.
     */
    function process_taxonomies() {
        foreach ( $this->objects['taxonomies'] as $taxonomy ) {
            $this->process_taxonomy( $taxonomy );
        }
    }


    /**
     * Process an individual taxonomy
     * 
     * @param string $taxonomy
     */
    function process_taxonomy( string $taxonomy ) {
        $terms = get_terms([
            'taxonomy' => $taxonomy,
            'hide_empty' => false
        ]);

        if ( !empty( $terms ) ) {
            $progress = WP_CLI\Utils\make_progress_bar( "Processing $taxonomy", count( $terms ) );

            foreach ( $terms as $term ) {
                $progress->tick();

                $flexi = get_field( raptor_get_flexi_field_key(), $term, false );

                if ( !$flexi ) {
                    continue;
                }

                foreach ( $flexi as &$flexi_block ) {
                    $flexi_block = call_user_func_array( $this->callback, [ $flexi_block, $term ] );
                }

                update_field( raptor_get_flexi_field_key(), array_filter( $flexi ), $term );
            }

            $progress->finish();
        } else {
            WP_CLI::warning( "Nothing found in the `$taxonomy` taxonomy." );
        }
    }
}
